clear;
clc;
load("Fitted_Parameters.mat");

%% ------------------------------------------
             %%% DEFINIZIONE DEI PARAMETRI INIZIALI %%%
    Tref  = 298.15;
    Kbolz = 1.380649e-23;
    q = 1.602176634e-19;
    Ki    = 1.3e-3;
    Eg    = 1.12;    
    A_cm2 = 1.44*10;     
    A     = A_cm2*1e-4;
    Isc   = 1.18e-7;
    Voc   = 7.00e-1;
    Top  = Tref;

%% ---------------------------------------------
              %%% DEFINIZIONE DEI PARAMETRI DI SIMULAZIONE %%%

% --- Profilo di G(t)
h = 3600;    % Assegnazione alla varibile 'h' dell'equivalente in secondi di un'ora
T_sim = 24*3;  % Tempo di simulazione in ore

% Profilo di G standard
%{
G_vals = [0, 0, 3, 3, 0, 0, 3, 3, 0, 0]';  % Valori assegnati all'irraggiamento in dei momenti precisi della simulazione
G_time = [0, (8*h)-1, 8*h, (18*h)-1, 18*h, (32*h)-1, 32*h, 42*h, (42*h)+1, T_sim*h]';  % Momenti della simulazione in cui viene cambiito il valore di irraggimento
%}

%{
% Profilo di G per simulazioni 2 Giorni
G_vals = [0, 0, 0, 3, 3, 0, 0, 3, 3, 0, 0]';  % Valori assegnati all'irraggiamento in dei momenti precisi della simulazione
G_time = [0, 1e-3, (8*h)-1, 8*h, (18*h)-1, 18*h, (32*h)-1, 32*h, 42*h, (42*h)+1, T_sim*h]';  % Momenti della simulazione in cui viene cambiito il valore di irraggimento
%}
%
% Profilo di G per simulazioni 3 Giorni
G_vals = [0, 0, 0, 3, 3, 0, 0, 3, 3, 0, 0, 3, 3, 0, 0]';  % Valori assegnati all'irraggiamento in dei momenti precisi della simulazione
G_time = [0, 1, (8*h)-1, 8*h, (18*h)-1, 18*h, (32*h)-1, 32*h, 42*h, (42*h)+1, (56*h)-1, 56*h, 66*h, (66*h)+1, T_sim*h]';  % Momenti della simulazione in cui viene cambiito il valore di irraggimento
%
%G_vals = [0, 0, 0, 0]';
%G_time = [0, 10, 100, 3600]';

% --- PV (SI)
NsPV  = 5; % Numero di celle fotovoltaiche in serie, aumenta tensione stessa corrente
NpPV  = 1; % Numero di celle fotovoltaiche in parallelo, aumenta corrente stessa tensione

% --- SuperCap (SI)
NpSC = 1; % Numero di supercondensatori in parallelo, tensione e carica massima invariata ma somma delle capacità
NsSC = 1; % Numero di supercondensatori in serie, tensione e carica massima aumenta ma capacità divisa

% --- Parametri calibrati del supercap (SI) --- 
% ----------------------------------------------------------------------------------------
%        ***** Copiare e incollare qui sotto i parametri calibrati del supercap ***** 
%                      ***** ottenuti da: Calibrazione_supercap.m *****
                           % C_need [F] - R1,R2 [Ω] - C0,C1,C2 [C]
C_need = 0.408;
R1 = 62.3;
R2 = 7789.2;
C0 = 0.279;
Cv = 0.067;
C2 = 0.019;

% ----------------------------------------------------------------------------------------

% --- Carico (SI)
Isens = 4e-6; % Corrente assorbita richiesta dal carico
VSensMax = 3.3; % Massimo valore di tensione accettato per il funzionamento del carico
VSensMin = 1.8; % Minimo valore di tensione richiesto per il funzionamento del carico

% --- Leakage (SI)
Rf = 500e3;
I_leak_max_C = (C_need*VSensMax)/(C_need*Rf);
I_leak_constant = 0;
Idark_constant =0;
%---------------------------------------------------------------------------
% Lettura dal file della corrente richiesta del carico
T = readtable('Current consumption_mod.txt', ...
    'Delimiter', '\t', ...
    'ReadVariableNames', true, ...
    'VariableNamingRule', 'preserve');

t_bp = T.("Time(s)");
i_bp = (T.("Current(mA)") * 1e-3);%-0.393e-3;

Tpat = t_bp(end);

if t_bp(end) > t_bp(1)
    t_bp = [t_bp; Tpat];
    i_bp = [i_bp; i_bp(1)];
end

% (Opzionale) rimuovi eventuali duplicati sul bordo
[ t_bp, idx ] = unique(t_bp,'stable');
i_bp = i_bp(idx);

% Crea timeseries
Isens_ts = timeseries(i_bp, t_bp);
Isens_ts.Name = 'Isens';

%--------------------------------------------------------------------------------------------

U0   = 0;                 % tensione iniziale desiderata
Q0   = C0*U0 + 0.5*Cv*U0^2;
Qmax = C0*VSensMax + 0.5*Cv*VSensMax^2;

           %%% FINE DEFINIZIONE DEI PARAMETRI DI SIMULAZIONE %%%

Autonomy = (C_need*(VSensMax-VSensMin))/((Isens+I_leak_max_C)*h)*0.686; 


% Calcolo RMS
i_rms = sqrt(mean(i_bp.^2));

% Conversione in microampere
i_micro = i_bp * 1e6;
i_rms_micro = i_rms * 1e6;
%{
% Plot
figure;
plot(t_bp, i_micro, 'b', 'LineWidth', 1.5); hold on;
yline(i_rms_micro, 'r--', 'LineWidth', 2);

xlabel('Tempo [s]');
ylabel('Corrente [µA]');
title('Corrente e valore quadratico medio (RMS)');
legend('Segnale originale', sprintf('RMS = %.3f µA', i_rms_micro), 'Location', 'best');
grid on;
%}

% --- Simulazione del modello e plot dei risultati

open_system(fullfile(pwd,'PV_SC_Model_SIM.slx'));
out = sim('PV_SC_Model_SIM');  

plot_IinSC_VinSC(out, G_time, G_vals, C_need, Autonomy, VSensMin, VSensMax, Isens, NpSC, NsSC, NpPV, NsPV, T_sim);

function plot_IinSC_VinSC(out, G_time, G_vals, C_need, ~, VSensMin, VSensMax, Isens, NpSC, NsSC, NpPV, NsPV, T_sim)


    % Estrai i timeseries
    tsI = out.IinSC;      % corrente
    tsV = out.VinSC;      % tensione
    tsS = out.Isens_scope;      % corrente

    % Tempo e dati
    t  = tsI.Time;
    yI = tsI.Data*1e6;    % µA
    yV = tsV.Data;        % V
    yS = tsS.Data*1e6;    % µA

    % Plot con doppio asse Y
    figure; hold on;
    yyaxis left
    hI = plot(t, yI, 'b-', 'LineWidth', 1.5);
    ylabel('IinSC [\muA]');
    %
    yyaxis left
    %hS = plot(t-850, yS, 'm-', 'LineWidth', 1.5);
    %----
    offset = 0;                     
    tq = t + offset;                   
    % Clampa dentro [t(1), t(end)] per evitare extrapolazioni e mantenere la curva
    tq = min(max(tq, t(1)), t(end));
    yS_shift = interp1(t, yS, tq, 'previous');
    hS = plot(t, yS_shift, 'm-', 'LineWidth', 1.5);
    %---
    ylabel('Iload [\muA]');
    %
    yyaxis right
    hV = plot(t, yV, 'r-', 'LineWidth', 1.5);
    ylabel('VinSC [V]');
    
    T = G_time(end) - G_time(1);                         
    t_wrap = mod(t - G_time(1), T) + G_time(1);          
    G_cont = interp1(G_time, G_vals, t_wrap, 'linear');  
    
    yyaxis right
    hG = plot(t, G_cont, 'g--', 'LineWidth', 1.5, 'DisplayName','G');

    xlabel('Tempo [ore]');
    xt = 0:1:T_sim;                % tick ogni ora
    xt_sec = xt*3600;              % corrispondenti in secondi

    set(gca, 'XTick', xt_sec, ...  % posizioni dei tick
         'XTickLabel', xt);        % etichette in ore

    grid on;
    legend([hI hV hG hS], {'IinSC','VinSC','G','Iload'}, 'Location','best');
    %legend([hI hV hG], {'IinSC','VinSC','G'}, 'Location','best');
    title('Profilo di irraggiamento (G), tensione (VinSC) e corrente (IinSC) in ingresso al supercap in funzione del tempo');
    subtitle({
    %sprintf('Capacità del supercap = %.3f [F], Autonomia stimata = %.1f [ore]', C_need, Autonomy)
    sprintf('Capacità del supercap = %.3f [F]', C_need)
    sprintf('Dinamica del carico %.1f-%.1f [V], Corrente richiesta dal carico = %.1f [µA]', VSensMin, VSensMax, Isens*1e6)
    sprintf('%d PV in parallelo, %d PV in serie, %d SC in parallelo, %d SC in serie', NpPV, NsPV, NpSC, NsSC)
    });
    
end
%